import React from 'react';
import { AvatarProps } from './components/Avatar';
import { IconProps } from './components/Icon';
import { ThumbnailProps } from './components/Thumbnail';
export interface OptionDescriptor {
    /** Value of the option */
    value: string;
    /** Display label for the option */
    label: React.ReactNode;
    /** Whether the option is disabled or not */
    disabled?: boolean;
    /** Whether the option is active or not */
    active?: boolean;
    /** Unique identifier for the option */
    id?: string;
    /** Media to display to the left of the option content */
    media?: React.ReactElement<IconProps | ThumbnailProps | AvatarProps>;
}
export interface SectionDescriptor {
    /** Collection of options within the section */
    options: OptionDescriptor[];
    /** Section title */
    title?: string;
}
export declare type Descriptor = SectionDescriptor | OptionDescriptor;
export declare type IconSource = React.FunctionComponent<React.SVGProps<SVGSVGElement>> | 'placeholder' | string;
export declare type HeadingTagName = 'h1' | 'h2' | 'h3' | 'h4' | 'h5' | 'h6' | 'p';
export declare type Target = '_blank' | '_self' | '_parent' | '_top';
export declare type Error = string | React.ReactElement | (string | React.ReactElement)[];
export interface BaseButton {
    /** A unique identifier for the button */
    id?: string;
    /** A destination to link to, rendered in the href attribute of a link */
    url?: string;
    /** Forces url to open in a new tab */
    external?: boolean;
    /** Where to display the url */
    target?: Target;
    /** Tells the browser to download the url instead of opening it. Provides a hint for the downloaded filename if it is a string value */
    download?: string | boolean;
    /** Allows the button to submit a form */
    submit?: boolean;
    /** Disables the button, disallowing merchant interaction */
    disabled?: boolean;
    /** Replaces button text with a spinner while a background action is being performed */
    loading?: boolean;
    /** Sets the button in a pressed state */
    pressed?: boolean;
    /** Visually hidden text for screen readers */
    accessibilityLabel?: string;
    /** A valid WAI-ARIA role to define the semantic value of this element */
    role?: string;
    /** Id of the element the button controls */
    ariaControls?: string;
    /** Tells screen reader the controlled element is expanded */
    ariaExpanded?: boolean;
    /** Indicates the ID of the element that describes the button */
    ariaDescribedBy?: string;
    /** Indicates the current checked state of the button when acting as a toggle or switch */
    ariaChecked?: 'false' | 'true';
    /** Callback when clicked */
    onClick?(): unknown;
    /** Callback when button becomes focussed */
    onFocus?(): void;
    /** Callback when focus leaves button */
    onBlur?(): void;
    /** Callback when a keypress event is registered on the button */
    onKeyPress?(event: React.KeyboardEvent<HTMLButtonElement>): void;
    /** Callback when a keyup event is registered on the button */
    onKeyUp?(event: React.KeyboardEvent<HTMLButtonElement>): void;
    /** Callback when a keydown event is registered on the button */
    onKeyDown?(event: React.KeyboardEvent<HTMLButtonElement>): void;
    /** Callback when mouse enter */
    onMouseEnter?(): void;
    /** Callback when element is touched */
    onTouchStart?(): void;
    /** Callback when pointerdown event is being triggered */
    onPointerDown?(): void;
}
export interface Action {
    /** A unique identifier for the action */
    id?: string;
    /** Content the action displays */
    content?: string;
    /** Visually hidden text for screen readers */
    accessibilityLabel?: string;
    /** A destination to link to, rendered in the action */
    url?: string;
    /** Forces url to open in a new tab */
    external?: boolean;
    /** Where to display the url */
    target?: Target;
    /** Callback when an action takes place */
    onAction?(): void;
    /** Callback when mouse enter */
    onMouseEnter?(): void;
    /** Callback when element is touched */
    onTouchStart?(): void;
}
export interface LinkAction {
    /** A unique identifier for the action */
    id?: string;
    /** Content the action displays */
    content?: string;
    /** Visually hidden text for screen readers */
    accessibilityLabel?: string;
    /** A destination to link to */
    url: string;
}
export interface BadgeAction {
    badge?: {
        status: 'new';
        content: string;
    };
}
export interface BaseCallbackAction {
    /** A unique identifier for the action */
    id?: string;
    /** Content the action displays */
    content?: string;
    /** Visually hidden text for screen readers */
    accessibilityLabel?: string;
    /** Callback when an action takes place */
    onAction(): void;
}
export interface CallbackAction extends BaseCallbackAction {
}
export interface DisableableAction extends Action {
    /** Whether or not the action is disabled */
    disabled?: boolean;
}
export interface DestructableAction extends Action {
    /** Destructive action */
    destructive?: boolean;
}
export interface IconableAction extends Action {
    /** Source of the icon */
    icon?: IconSource;
}
export interface LoadableAction extends Action {
    /** Should a spinner be displayed */
    loading?: boolean;
}
export interface OutlineableAction extends Action {
    /** Should action be displayed as an outlined button */
    outline?: boolean;
}
export interface PlainAction extends Action {
    /** Should action be displayed as a plain link */
    plain?: boolean;
}
export interface TooltipAction {
    /** Text content to render in a tooltip */
    helpText?: React.ReactNode;
}
export interface ActionListItemDescriptor extends DisableableAction, DestructableAction {
    /** Visually hidden text for screen readers */
    accessibilityLabel?: string;
    /** @deprecated Badge component */
    badge?: {
        status: 'new';
        content: string;
    };
    /** Additional hint text to display with item */
    helpText?: React.ReactNode;
    /** @deprecated Source of the icon */
    icon?: IconSource;
    /** @deprecated Image source */
    image?: string;
    /** Prefix source */
    prefix?: React.ReactNode;
    /** Suffix source */
    suffix?: React.ReactNode;
    /** @deprecated Add an ellipsis suffix to action content. ellipsis appends `...` without truncating. Use truncate instead. */
    ellipsis?: boolean;
    /** Truncate the action content either at the beginning or at the end */
    truncate?: boolean;
    /** Whether the action is active or not */
    active?: boolean;
    /** Defines a role for the action */
    role?: string;
}
export interface ActionListSection {
    /** Section title */
    title?: string | React.ReactNode;
    /** Collection of action items for the list */
    items: readonly ActionListItemDescriptor[];
}
export interface ComplexAction extends Action, DisableableAction, DestructableAction, IconableAction, OutlineableAction, LoadableAction, PlainAction {
}
export interface MenuActionDescriptor extends ComplexAction, TooltipAction {
    /** Zero-indexed numerical position. Overrides the action's order in the menu */
    index?: number;
}
export interface MenuGroupDescriptor extends BadgeAction {
    /** Menu group title */
    title: string;
    /** List of actions */
    actions: ActionListItemDescriptor[];
    /** Icon to display */
    icon?: IconableAction['icon'];
    /** Action details */
    details?: React.ReactNode;
    /** Disables action button */
    disabled?: boolean;
    /** Zero-indexed numerical position. Overrides the group's order in the menu. */
    index?: number;
    /** Callback when any action takes place */
    onActionAnyItem?: ActionListItemDescriptor['onAction'];
    /** Callback when the menu is clicked */
    onClick?(openActions: () => void): void;
}
export interface ConnectedDisclosure {
    /** Visually hidden label for the connected disclosure button.
     * @default 'Related actions'
     */
    accessibilityLabel?: string;
    /** Whether or not the disclosure is disabled */
    disabled?: boolean;
    /** List of actions */
    actions: ActionListItemDescriptor[];
}
export declare enum Key {
    Backspace = 8,
    Tab = 9,
    Enter = 13,
    Shift = 16,
    Ctrl = 17,
    Alt = 18,
    Pause = 19,
    CapsLock = 20,
    Escape = 27,
    Space = 32,
    PageUp = 33,
    PageDown = 34,
    End = 35,
    Home = 36,
    LeftArrow = 37,
    UpArrow = 38,
    RightArrow = 39,
    DownArrow = 40,
    Insert = 45,
    Delete = 46,
    Key0 = 48,
    Key1 = 49,
    Key2 = 50,
    Key3 = 51,
    Key4 = 52,
    Key5 = 53,
    Key6 = 54,
    Key7 = 55,
    Key8 = 56,
    Key9 = 57,
    KeyA = 65,
    KeyB = 66,
    KeyC = 67,
    KeyD = 68,
    KeyE = 69,
    KeyF = 70,
    KeyG = 71,
    KeyH = 72,
    KeyI = 73,
    KeyJ = 74,
    KeyK = 75,
    KeyL = 76,
    KeyM = 77,
    KeyN = 78,
    KeyO = 79,
    KeyP = 80,
    KeyQ = 81,
    KeyR = 82,
    KeyS = 83,
    KeyT = 84,
    KeyU = 85,
    KeyV = 86,
    KeyW = 87,
    KeyX = 88,
    KeyY = 89,
    KeyZ = 90,
    LeftMeta = 91,
    RightMeta = 92,
    Select = 93,
    Numpad0 = 96,
    Numpad1 = 97,
    Numpad2 = 98,
    Numpad3 = 99,
    Numpad4 = 100,
    Numpad5 = 101,
    Numpad6 = 102,
    Numpad7 = 103,
    Numpad8 = 104,
    Numpad9 = 105,
    Multiply = 106,
    Add = 107,
    Subtract = 109,
    Decimal = 110,
    Divide = 111,
    F1 = 112,
    F2 = 113,
    F3 = 114,
    F4 = 115,
    F5 = 116,
    F6 = 117,
    F7 = 118,
    F8 = 119,
    F9 = 120,
    F10 = 121,
    F11 = 122,
    F12 = 123,
    NumLock = 144,
    ScrollLock = 145,
    Semicolon = 186,
    Equals = 187,
    Comma = 188,
    Dash = 189,
    Period = 190,
    ForwardSlash = 191,
    GraveAccent = 192,
    OpenBracket = 219,
    BackSlash = 220,
    CloseBracket = 221,
    SingleQuote = 222
}
export interface CheckboxHandles {
    focus(): void;
}
export declare type NonEmptyArray<T> = [
    T,
    ...T[]
];
export declare type ArrayElement<T> = T extends (infer U)[] ? U : never;
export interface AppliedFilterInterface {
    /** A unique key used to identify the applied filter */
    key: string;
    /** A label for the applied filter */
    label: string;
    /** Callback when the remove button is pressed */
    onRemove(key: string): void;
}
export interface FilterInterface {
    /** A unique key used to identify the filter */
    key: string;
    /** The label for the filter */
    label: string;
    /** The markup for the given filter */
    filter: React.ReactNode;
    /** Whether or not the filter should have a shortcut popover displayed */
    shortcut?: boolean;
    /** Whether or not the filter should be pinned, permanently displaying the filter */
    pinned?: boolean;
    /** Whether or not the filter is disabled */
    disabled?: boolean;
    /**
     * @default false
     * Whether or not the clear button is displayed
     */
    hideClearButton?: boolean;
    /** Optional callback when filter is pressed */
    onAction?: () => void;
    /** Suffix source */
    suffix?: React.ReactNode;
    /** Optional section heading that this filter will go under  */
    section?: string;
}
//# sourceMappingURL=types.d.ts.map
