"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chalk = require("chalk");
const core_1 = require("@oclif/core");
const plugins_1 = require("../../plugins");
const util_1 = require("../../util");
class PluginsIndex extends core_1.Command {
    constructor() {
        super(...arguments);
        this.plugins = new plugins_1.default(this.config);
    }
    async run() {
        var _a;
        const { flags } = await this.parse(PluginsIndex);
        let plugins = this.config.getPluginsList();
        (0, util_1.sortBy)(plugins, p => this.plugins.friendlyName(p.name));
        if (!flags.core) {
            plugins = plugins.filter(p => p.type !== 'core' && p.type !== 'dev');
        }
        if (plugins.length === 0)
            this.log('No plugins installed.');
        const results = this.config.getPluginsList();
        const userAndLinkedPlugins = new Set(results.filter(p => p.type === 'user' || p.type === 'link').map(p => p.name));
        const jitPluginsConfig = (_a = this.config.pjson.oclif.jitPlugins) !== null && _a !== void 0 ? _a : {};
        const jitPlugins = Object.entries(jitPluginsConfig)
            .map(([name, version]) => ({ name, version, type: 'jit' }))
            .filter(p => !userAndLinkedPlugins.has(p.name));
        (0, util_1.sortBy)(jitPlugins, p => p.name);
        if (!this.jsonEnabled()) {
            this.display(plugins);
            this.displayJitPlugins(jitPlugins);
        }
        return [...results, ...jitPlugins];
    }
    display(plugins) {
        for (const plugin of plugins.filter((p) => !p.parent)) {
            this.log(this.formatPlugin(plugin));
            if (plugin.children && plugin.children.length > 0) {
                const tree = this.createTree(plugin);
                tree.display();
            }
        }
    }
    displayJitPlugins(jitPlugins) {
        if (jitPlugins.length === 0)
            return;
        this.log(chalk.dim('\nUninstalled JIT Plugins:'));
        for (const { name, version } of jitPlugins) {
            this.log(`${this.plugins.friendlyName(name)} ${chalk.dim(version)}`);
        }
    }
    createTree(plugin) {
        const tree = core_1.ux.tree();
        for (const p of plugin.children) {
            const name = this.formatPlugin(p);
            tree.insert(name, this.createTree(p));
        }
        return tree;
    }
    formatPlugin(plugin) {
        let output = `${this.plugins.friendlyName(plugin.name)} ${chalk.dim(plugin.version)}`;
        if (plugin.type !== 'user')
            output += chalk.dim(` (${plugin.type})`);
        if (plugin.type === 'link')
            output += ` ${plugin.root}`;
        else if (plugin.tag && plugin.tag !== 'latest')
            output += chalk.dim(` (${String(plugin.tag)})`);
        return output;
    }
}
exports.default = PluginsIndex;
PluginsIndex.enableJsonFlag = true;
PluginsIndex.flags = {
    core: core_1.Flags.boolean({ description: 'Show core plugins.' }),
};
PluginsIndex.description = 'List installed plugins.';
PluginsIndex.examples = ['$ <%- config.bin %> plugins'];
