/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { toAnyValue, toAttributes, toKeyValue } from '../common/internal';
import { hexToBase64, hrTimeToNanoseconds } from '@opentelemetry/core';
export function createExportLogsServiceRequest(logRecords, useHex) {
    return {
        resourceLogs: logRecordsToResourceLogs(logRecords, useHex),
    };
}
function createResourceMap(logRecords) {
    const resourceMap = new Map();
    for (const record of logRecords) {
        const { resource, instrumentationScope: { name, version = '', schemaUrl = '' }, } = record;
        let ismMap = resourceMap.get(resource);
        if (!ismMap) {
            ismMap = new Map();
            resourceMap.set(resource, ismMap);
        }
        const ismKey = `${name}@${version}:${schemaUrl}`;
        let records = ismMap.get(ismKey);
        if (!records) {
            records = [];
            ismMap.set(ismKey, records);
        }
        records.push(record);
    }
    return resourceMap;
}
function logRecordsToResourceLogs(logRecords, useHex) {
    const resourceMap = createResourceMap(logRecords);
    return Array.from(resourceMap, ([resource, ismMap]) => ({
        resource: {
            attributes: toAttributes(resource.attributes),
            droppedAttributesCount: 0,
        },
        scopeLogs: Array.from(ismMap, ([, scopeLogs]) => {
            const { instrumentationScope: { name, version, schemaUrl }, } = scopeLogs[0];
            return {
                scope: { name, version },
                logRecords: scopeLogs.map(log => toLogRecord(log, useHex)),
                schemaUrl,
            };
        }),
        schemaUrl: undefined,
    }));
}
function toLogRecord(log, useHex) {
    var _a, _b, _c, _d, _e;
    return {
        timeUnixNano: hrTimeToNanoseconds(log.hrTime),
        observedTimeUnixNano: hrTimeToNanoseconds(log.hrTimeObserved),
        severityNumber: toSeverityNumber(log.severityNumber),
        severityText: log.severityText,
        body: toAnyValue(log.body),
        attributes: toLogAttributes(log.attributes),
        droppedAttributesCount: 0,
        flags: (_a = log.spanContext) === null || _a === void 0 ? void 0 : _a.traceFlags,
        traceId: useHex
            ? (_b = log.spanContext) === null || _b === void 0 ? void 0 : _b.traceId
            : optionalHexToBase64((_c = log.spanContext) === null || _c === void 0 ? void 0 : _c.traceId),
        spanId: useHex
            ? (_d = log.spanContext) === null || _d === void 0 ? void 0 : _d.spanId
            : optionalHexToBase64((_e = log.spanContext) === null || _e === void 0 ? void 0 : _e.spanId),
    };
}
function toSeverityNumber(severityNumber) {
    return severityNumber;
}
function optionalHexToBase64(str) {
    if (str === undefined)
        return undefined;
    return hexToBase64(str);
}
export function toLogAttributes(attributes) {
    return Object.keys(attributes).map(key => toKeyValue(key, attributes[key]));
}
//# sourceMappingURL=index.js.map