import { AnyAction, Group, MetaAction, Dispatch, ErrorSubscriber, ActionCallback, Unsubscribe, ActionSetInterface, ActionSetOptions } from '../actions/types';
import type { Context } from '../MessageTransport';
/**
 * @internal
 */
export declare enum MessageType {
    GetState = "getState",
    Dispatch = "dispatch",
    Subscribe = "subscribe",
    Unsubscribe = "unsubscribe"
}
/**
 * @public
 */
export interface AppConfigV2 {
    apiKey: string;
    host: string;
    forceRedirect?: boolean;
}
/**
 * @internal
 */
export interface TransportDispatch {
    type: MessageType;
    source: AppConfigV2;
    payload?: AnyAction;
}
/**
 * @internal
 */
export interface Dispatcher {
    (type: MessageType, payload?: AnyAction | undefined): void;
}
export interface StaffMemberDetails {
    accountAccess: string;
    locale: string;
}
/**
 * @internal
 */
export declare enum PermissionType {
    Dispatch = "Dispatch",
    Subscribe = "Subscribe"
}
/**
 * @internal
 */
export declare type FeaturesAvailable<T extends Group = Group> = Record<T, FeaturesAction>;
/**
 * @internal
 */
export declare type FeaturesState<T = FeaturesAvailable> = {
    [key in Context]?: T;
};
/**
 * @internal
 */
export declare type FeaturePermission = {
    [key in PermissionType]: boolean;
};
/**
 * @internal
 */
export interface FeaturesAction {
    [key: string]: FeaturePermission;
}
/**
 * @internal
 */
export declare type StateWithoutFeatures<S> = Omit<S, 'features'>;
/**
 * The shared App Bridge state which contains information about
 * the current staff member, application context, features, etc...
 * @public
 */
export interface AppBridgeState {
    staffMember: StaffMemberDetails;
    context: Context;
    features: FeaturesAvailable;
}
declare type PathImpl<T, Key extends keyof T> = Key extends string ? T[Key] extends Record<string, any> ? `${Key}.${PathImpl<T[Key], Exclude<keyof T[Key], keyof any[]>> & string}` | `${Key}.${Exclude<keyof T[Key], keyof any[]> & string}` : never : never;
declare type PathImpl2<T> = PathImpl<T, keyof T> | keyof T;
/**
 * @internal
 */
export declare type Path<T> = PathImpl2<T> extends string | keyof T ? PathImpl2<T> : keyof T;
/**
 * @internal
 */
export declare type PathValue<T, P extends Path<T>> = P extends `${infer Key}.${infer Rest}` ? Key extends keyof T ? Rest extends Path<T[Key]> ? PathValue<T[Key], Rest> : never : never : P extends keyof T ? T[P] : never;
/**
 * Application instance, required for use with actions.
 * @public
 * @remarks
 * You should be using this as a static type. Currently this interface is a generic but it will set to a static type
 * in the future.
 */
export interface ClientApplication<S extends AppBridgeState = AppBridgeState> {
    dispatch: Dispatch<AnyAction>;
    localOrigin: string;
    hostOrigin: string;
    error: ErrorSubscriber;
    hooks?: HooksInterface;
    getState(): Promise<S>;
    getState<K extends keyof S>(query: K): Promise<S[K]>;
    getState<PartialState>(query: keyof S): Promise<PartialState>;
    getState<K extends Path<StateWithoutFeatures<S>>>(query: K): Promise<PathValue<StateWithoutFeatures<S>, K>>;
    featuresAvailable(): Promise<FeaturesAvailable<Group>>;
    featuresAvailable<T extends Group[]>(...features: T): Promise<FeaturesAvailable<(typeof features)[number]>>;
    subscribe(callback: ActionCallback): Unsubscribe;
    subscribe(eventNameSpace: string, callback: ActionCallback, id?: string): Unsubscribe;
}
/**
 *
 * There are two types of hooks: `LifecycleHook.DispatchAction` and `LifecycleHook.UpdateAction`.
 *
 * @remarks
 * `DispatchAction` hooks are run when an action is dispatched with the `.dispatch()` function:
 *
 * ```js
 * const toastNotice = Toast.create(app, {message: 'Product saved'});
 * toastNotice.dispatch(Toast.Action.SHOW);
 * ```
 *
 * `UpdateAction` hooks are run when an action is updated, using the `.set()` function:
 *
 * ```js
 * toastNotice.set({message: 'Product could not be saved', isError: true});
 * ```
 *
 * @public
 */
export declare enum LifecycleHook {
    UpdateAction = "UpdateAction",
    DispatchAction = "DispatchAction"
}
/**
 * @internal
 */
export interface UpdateActionHandler {
    <O>(this: ActionSetInterface & ActionSetOptions<O>, options: O): any;
}
/**
 * @internal
 */
export interface UpdateActionHook {
    (next: Function): UpdateActionHandler;
}
/**
 * @internal
 */
export interface DispatchActionHandler {
    <A extends MetaAction>(this: ClientApplication, action: A): any;
}
/**
 * @internal
 */
export interface DispatchActionHook {
    (next: Function): DispatchActionHandler;
}
/**
 * @internal
 */
export interface LifecycleHandler {
    (next: Function): (...args: any[]) => any;
}
/**
 * @internal
 */
export interface HooksInterface {
    set(hook: LifecycleHook.UpdateAction, handler: UpdateActionHook): any;
    set(hook: LifecycleHook.DispatchAction, handler: DispatchActionHook): any;
    set(hook: LifecycleHook, handler: LifecycleHandler): any;
    get(hook: LifecycleHook): LifecycleHandler[] | undefined;
    run<C>(hook: LifecycleHook, final: Function, context: C, ...arg: any[]): any;
}
/**
 * @internal
 */
export interface TransportSubscription {
    id?: string;
    type: string;
}
export {};
