# `@shopify/app-bridge-react`

**[Join our team and work on libraries like this one.](https://www.shopify.ca/careers)**

[![License: MIT](https://img.shields.io/badge/License-MIT-green.svg)](LICENSE.md)
[![npm version](https://badge.fury.io/js/%40shopify%2Fapp-bridge-react.svg)](https://badge.fury.io/js/%40shopify%2Fapp-bridge-react.svg)
[![npm bundle size (minified + gzip)](https://img.shields.io/bundlephobia/minzip/@shopify/app-bridge-react.svg)](https://img.shields.io/bundlephobia/minzip/@shopify/app-bridge-react.svg)

Shopify App Bridge offers React component wrappers for some App Bridge actions. This is a great option if you are already using React and want to follow familiar patterns.

Instead of using App Bridge actions directly:

```js
import createApp from '@shopify/app-bridge';
import * as TitleBar from '@shopify/app-bridge/actions/TitleBar';

const app = createApp({
  apiKey: 'API key from Shopify Partner Dashboard',
  host: 'host from URL search parameter',
});

const titleBarOptions = {
  title: 'My page title',
};

const myTitleBar = TitleBar.create(app, titleBarOptions);
```

Use the React component wrappers:

> **Note**: only one `Provider` is needed for your application.

```jsx
// MyApp.jsx
import React from 'react';
import ReactDOM from 'react-dom/client';
import {Provider} from '@shopify/app-bridge-react';
import {RedirectButton} from './RedirectButton';

function MyApp() {
  const config = {
    apiKey: 'API key from Shopify Partner Dashboard',
    host: 'host from URL search parameter',
  };

  return (
    <Provider config={config}>
      <RedirectButton />
    </Provider>
  );
}

const root = document.createElement('div');
document.body.appendChild(root);
ReactDOM.createRoot(root).render(<MyApp />);
```

To access an `app` from the Provider, use the `useAppBridge` hook (since `v1.25.0`):

```jsx
// RedirectButton.jsx
import {Button, Redirect} from '@shopify/app-bridge/actions';
import {useAppBridge} from '@shopify/app-bridge-react';

function RedirectButton() {
  const app = useAppBridge();
  const redirect = Redirect.create(app);

  const handleClick = () => {
    // Go to {shopUrl}/admin/customers with newContext
    redirect.dispatch(Redirect.Action.ADMIN_PATH, {
      path: '/customers',
      newContext: true,
    });
  };

  return <Button onClick={handleClick}>Activate Lasers for Customers</Button>;
}
```

## Migrating from 2.x.x to 3.0.0

There are two breaking API changes in version 3.0.0.

- `useContextualSaveBar`
- `useToast`

### useContextualSaveBar

In version 2.x.x, `useContextualSaveBar` was called with all of its options. It then created the contextual save bar and dispatched show and hide functions as an internal side effect based on the `visible` prop and any changes in options. It did not return anything.

#### Example code

```tsx
useContextualSaveBar({
  discardAction,
  saveAction,
  fullWidth,
  leaveConfirmationDisable,
  visible,
});
```

In version 3.0.0, `useContextualSaveBar` becomes more declarative. The hook is called without any props and it returns several objects and methods that can be used to update contextual save bar options, as well as to show and hide the component.

#### Example code

```tsx
const {show, hide, saveAction, discardAction} = useContextualSaveBar();
const fullWidth = true;
const leaveConfirmationDisabled = false;

<Button
  variant="primary"
  onClick={() => {
    show(fullWidth, leaveConfirmationDisabled);
    discardAction.setOptions({onAction: () => console.log('On discard')});
    saveAction.setOptions({onAction: () => console.log('On save')});
  }}
>
  Show ContextualSaveBar
</Button>
<Button onClick={hide}>Hide ContextualSaveBar</Button>
```

See [useContextualSaveBar docs](https://shopify.dev/apps/tools/app-bridge/react-components/contextual-save-bar) for more details.

### useToast

In version 2.x.x, `useToast` returns a `show` method that accepts one prop - an object with the following properties:

- `content`
- `duration`
- `isError`
- `onDismiss`

All the props except `content` are optional.

#### Example code

```tsx
const {show} = useToast

show({
  content: 'Success!,
  duration: 2000,
  isError: false,
  onDismiss: () => console.log('Dismissed'),
})
```

In version 3.0.0, the `content` moves to a top-level prop and the remaining properties are passed in as an optional options prop (all properties in the options object remain optional).
`

#### Example code

```tsx
const {show} = useToast;

show('Success!', {
  duration: 2000,
  isError: false,
  onDismiss: () => console.log('Dismissed'),
});
```

See [useToast docs](https://shopify.dev/apps/tools/app-bridge/react-components/toast) for more details.

## Legacy support

If you are using App Bridge `v1.24.0` and below, or your project does not support React hooks, alternative methods on how to access `app` can be found in the [Provider docs](https://shopify.dev/tools/app-bridge/react-components/provider#accessing-the-app-bridge-client-directly).
