'use strict';

var constants = require('./constants.js');

function formatErrorMessage(message, client = constants.CLIENT) {
    return message.startsWith(`${client}`) ? message : `${client}: ${message}`;
}
function getErrorMessage(error) {
    return error instanceof Error ? error.message : JSON.stringify(error);
}
function getErrorCause(error) {
    return error instanceof Error && error.cause ? error.cause : undefined;
}
function combineErrors(dataArray) {
    return dataArray.flatMap(({ errors }) => {
        return errors ?? [];
    });
}
function validateRetries({ client, retries, }) {
    if (retries !== undefined &&
        (typeof retries !== "number" ||
            retries < constants.MIN_RETRIES ||
            retries > constants.MAX_RETRIES)) {
        throw new Error(`${client}: The provided "retries" value (${retries}) is invalid - it cannot be less than ${constants.MIN_RETRIES} or greater than ${constants.MAX_RETRIES}`);
    }
}
function getKeyValueIfValid(key, value) {
    return value &&
        (typeof value !== "object" ||
            Array.isArray(value) ||
            (typeof value === "object" && Object.keys(value).length > 0))
        ? { [key]: value }
        : {};
}
function buildDataObjectByPath(path, data) {
    if (path.length === 0) {
        return data;
    }
    const key = path.pop();
    const newData = {
        [key]: data,
    };
    if (path.length === 0) {
        return newData;
    }
    return buildDataObjectByPath(path, newData);
}
function combineObjects(baseObject, newObject) {
    return Object.keys(newObject || {}).reduce((acc, key) => {
        if ((typeof newObject[key] === "object" || Array.isArray(newObject[key])) &&
            baseObject[key]) {
            acc[key] = combineObjects(baseObject[key], newObject[key]);
            return acc;
        }
        acc[key] = newObject[key];
        return acc;
    }, Array.isArray(baseObject) ? [...baseObject] : { ...baseObject });
}
function buildCombinedDataObject([initialDatum, ...remainingData]) {
    return remainingData.reduce(combineObjects, { ...initialDatum });
}

exports.buildCombinedDataObject = buildCombinedDataObject;
exports.buildDataObjectByPath = buildDataObjectByPath;
exports.combineErrors = combineErrors;
exports.formatErrorMessage = formatErrorMessage;
exports.getErrorCause = getErrorCause;
exports.getErrorMessage = getErrorMessage;
exports.getKeyValueIfValid = getKeyValueIfValid;
exports.validateRetries = validateRetries;
//# sourceMappingURL=utilities.js.map
