'use strict';

var graphqlClient = require('./graphql-client/dist/graphql-client/graphql-client.js');
var validations = require('./graphql-client/dist/api-client-utilities/validations.js');
var apiVersions = require('./graphql-client/dist/api-client-utilities/api-versions.js');
var utilities = require('./graphql-client/dist/api-client-utilities/utilities.js');
var constants = require('./constants.js');
var validations$1 = require('./validations.js');

function createStorefrontApiClient({ storeDomain, apiVersion, publicAccessToken, privateAccessToken, clientName, retries = 0, customFetchApi, logger, }) {
    const currentSupportedApiVersions = apiVersions.getCurrentSupportedApiVersions();
    const storeUrl = validations.validateDomainAndGetStoreUrl({
        client: constants.CLIENT,
        storeDomain,
    });
    const baseApiVersionValidationParams = {
        client: constants.CLIENT,
        currentSupportedApiVersions,
        logger,
    };
    validations.validateApiVersion({ ...baseApiVersionValidationParams, apiVersion });
    validations$1.validateRequiredAccessTokens(publicAccessToken, privateAccessToken);
    validations$1.validatePrivateAccessTokenUsage(privateAccessToken);
    const apiUrlFormatter = generateApiUrlFormatter(storeUrl, apiVersion, baseApiVersionValidationParams);
    const config = {
        storeDomain: storeUrl,
        apiVersion,
        ...(publicAccessToken
            ? { publicAccessToken }
            : {
                privateAccessToken: privateAccessToken,
            }),
        headers: {
            "Content-Type": constants.DEFAULT_CONTENT_TYPE,
            Accept: constants.DEFAULT_CONTENT_TYPE,
            [constants.SDK_VARIANT_HEADER]: constants.DEFAULT_SDK_VARIANT,
            [constants.SDK_VERSION_HEADER]: constants.DEFAULT_CLIENT_VERSION,
            ...(clientName ? { [constants.SDK_VARIANT_SOURCE_HEADER]: clientName } : {}),
            ...(publicAccessToken
                ? { [constants.PUBLIC_ACCESS_TOKEN_HEADER]: publicAccessToken }
                : { [constants.PRIVATE_ACCESS_TOKEN_HEADER]: privateAccessToken }),
        },
        apiUrl: apiUrlFormatter(),
        clientName,
    };
    const graphqlClient$1 = graphqlClient.createGraphQLClient({
        headers: config.headers,
        url: config.apiUrl,
        retries,
        customFetchApi,
        logger,
    });
    const getHeaders = utilities.generateGetHeaders(config);
    const getApiUrl = generateGetApiUrl(config, apiUrlFormatter);
    const getGQLClientParams = utilities.generateGetGQLClientParams({
        getHeaders,
        getApiUrl,
    });
    const client = {
        config,
        getHeaders,
        getApiUrl,
        fetch: (...props) => {
            return graphqlClient$1.fetch(...getGQLClientParams(...props));
        },
        request: (...props) => {
            return graphqlClient$1.request(...getGQLClientParams(...props));
        },
        requestStream: (...props) => {
            return graphqlClient$1.requestStream(...getGQLClientParams(...props));
        },
    };
    return Object.freeze(client);
}
function generateApiUrlFormatter(storeUrl, defaultApiVersion, baseApiVersionValidationParams) {
    return (apiVersion) => {
        if (apiVersion) {
            validations.validateApiVersion({
                ...baseApiVersionValidationParams,
                apiVersion,
            });
        }
        const urlApiVersion = (apiVersion ?? defaultApiVersion).trim();
        return `${storeUrl}/api/${urlApiVersion}/graphql.json`;
    };
}
function generateGetApiUrl(config, apiUrlFormatter) {
    return (propApiVersion) => {
        return propApiVersion ? apiUrlFormatter(propApiVersion) : config.apiUrl;
    };
}

exports.createStorefrontApiClient = createStorefrontApiClient;
//# sourceMappingURL=storefront-api-client.js.map
