"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThemeLiquidDocsManager = void 0;
const node_path_1 = __importDefault(require("node:path"));
const promises_1 = __importDefault(require("node:fs/promises"));
const jsonSchemaCompiler_1 = require("./jsonSchemaCompiler");
const themeLiquidDocsDownloader_1 = require("./themeLiquidDocsDownloader");
const utils_1 = require("./utils");
const fp_1 = require("lodash/fp");
class ThemeLiquidDocsManager {
    constructor(log = utils_1.noop) {
        this.log = log;
        this.filters = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('filters'), JSON.parse, []);
        });
        this.objects = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('objects'), JSON.parse, []);
        });
        this.tags = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('tags'), JSON.parse, []);
        });
        this.systemTranslations = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('shopify_system_translations'), JSON.parse, {});
        });
        this.sectionSchema = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('section_schema'), identity, '{}');
        });
        this.validateSectionSchema = (0, utils_1.memo)(async () => {
            return findSuitableResource([this.sectionSchema], (0, fp_1.pipe)(JSON.parse, jsonSchemaCompiler_1.compileJsonSchema), alwaysValid());
        });
        this.translationSchema = (0, utils_1.memo)(async () => {
            return findSuitableResource(this.loaders('translations_schema'), identity, '{}');
        });
        /**
         * The setup method checks that the latest revision matches the one from
         * Shopify/theme-liquid-docs. If there's a diff in revision, it means
         * that the documentations that you have locally are out of date.
         *
         * The setup method then downloads the other files.
         */
        this.setup = (0, utils_1.memo)(async () => {
            try {
                if (!(await (0, themeLiquidDocsDownloader_1.exists)(utils_1.root))) {
                    await promises_1.default.mkdir(utils_1.root, { recursive: true });
                }
                const local = await this.latestRevision();
                await (0, utils_1.download)('latest');
                const remote = await this.latestRevision();
                if (local !== remote) {
                    await Promise.all(themeLiquidDocsDownloader_1.Resources.map((resource) => (0, utils_1.download)(resource)));
                }
            }
            catch (error) {
                if (error instanceof Error)
                    this.log(error.message);
                return;
            }
        });
    }
    async latestRevision() {
        var _a;
        const latest = await findSuitableResource([() => this.load('latest')], JSON.parse, {});
        return (_a = latest['revision']) !== null && _a !== void 0 ? _a : '';
    }
    async loadResource(name) {
        // Always wait for setup first. Since it's memoized, this will always
        // be the same promise.
        await this.setup();
        return this.load(name);
    }
    async load(name) {
        return promises_1.default.readFile((0, utils_1.filePath)(name), 'utf8');
    }
    loaders(name) {
        return [() => this.loadResource(name), () => fallback(name)];
    }
}
exports.ThemeLiquidDocsManager = ThemeLiquidDocsManager;
const identity = (x) => x;
/**
 * Find the first resource that can be loaded and transformed.
 *
 * Will try to load the resource from the loaders in order. If the loader
 * throws an error, it will try the next loader. If all loaders throw an
 * error, it will return the default value.
 *
 * This should allow us to load the latest version of the resource if it's
 * available, and fall back to the local version if it's not. If neither
 * work, we'll just return the default value.
 */
async function findSuitableResource(dataLoaders, transform, defaultValue) {
    for (const loader of dataLoaders) {
        try {
            return transform(await loader());
        }
        catch (_) {
            continue;
        }
    }
    return defaultValue;
}
/**
 * The root directory for the data files. This is different in the VS Code build
 * (since those files are copied to the dist folder at a different relative path)
 */
function dataRoot() {
    if (process.env.WEBPACK_MODE) {
        return node_path_1.default.resolve(__dirname, './data');
    }
    else {
        return node_path_1.default.resolve(__dirname, '../data');
    }
}
/** Returns the at-build-time path to the fallback data file. */
async function fallback(name) {
    return promises_1.default.readFile(node_path_1.default.resolve(dataRoot(), `${name}.json`), 'utf8');
}
function alwaysValid() {
    const validate = () => true;
    validate.errors = null;
    return validate;
}
//# sourceMappingURL=themeLiquidDocsManager.js.map